/* ****************************************************************************
 * Copyright: 2017-2025 RAYLASE GmbH
 * This source code is the proprietary confidential property of RAYLASE GmbH.
 * Reproduction, publication, or any form of distribution to
 * any party other than the licensee is strictly prohibited.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
 * IN THE SOFTWARE.
 */

#pragma once

#include "ClientTransport.h"
#include "Context.h"
#include "CustomRpcRemoteExceptionSerializer.h"
#include "InputPacket.h"
#include "OutputPacket.h"
#include "Rpc.h"

#include <chrono>
#include <cstdint>
#include <msgpack.hpp>
#include <optional>
#include <stdexcept>
#include <string>

class RpcClientConfiguration;

class RpcClient : public Rpc
{
private:
	std::uint32_t _messageId = 0;
	std::optional<ClientTransport> _transport;
	Context _Context;

public:
	RpcClient(RpcClientConfiguration& config);
	~RpcClient();

	const RpcClientConfiguration& GetConfig() const override { return _config; }
	void SetReceiveTimeout(std::chrono::milliseconds timeoutMs);
	void EnableLogging(std::string path, bool timestamp, bool append, std::int32_t truncateArgumentLength);
	void DisableLogging() noexcept;
	void AppendLogMessage(const char* type, std::string message, bool function = true);
	void AppendLogMessage(std::string message);

	void Start();
	void Stop() noexcept;

	bool IsConnected() const noexcept
	{
		if (_active && (_Context._Transport != nullptr))
			return _Context._Transport->IsConnected();

		return false;
	}

	template<typename T, typename... Args> T Call(std::string method, Args&&... args)
	{
		try
		{
			static const std::uint16_t size = sizeof...(Args);
			std::uint32_t id = _messageId++;
			if (!_active || !_Context._Transport->IsConnected())
				throw std::runtime_error("Connection not active!");
			if (_Context._Instance->GetConfig().EnableLogging)
			{
				_Context._Transport->_Logger.BeginLine("TX", method);
				_Context._Transport->_Logger.Append(args...);
				_Context._Transport->_Logger.EndLine();
			}
			{
				std::unique_ptr<OutputPacket> packet = OutputPacket::AllocateRequest(id);
				packet->Message.pack(method);
				packet->Message.pack_array(size);
				(..., packet->Message.pack(std::forward<Args>(args)));
				_transport->Connection.Write(packet->Close(), packet->Size());
			}

			std::unique_ptr<InputPacket> result = _transport->Read(id, GetConfig().ReceiveTimeoutMs);
			msgpack::object_handle o;
			result->Message.next(o);
			if ((std::int32_t)result->GetType() >= (std::int32_t)PacketType::Error)
			{
				RpcRemoteException r;
				o.get().convert<RpcRemoteException>(r);
				if (_Context._Instance->GetConfig().EnableLogging)
				{
					_Context._Transport->_Logger.BeginLine("RX", method);
					_Context._Transport->_Logger.Append(r);
					_Context._Transport->_Logger.EndLine();
				}
				throw r;
			}
			T val;
			o.get().convert<T>(val);
			if (_Context._Instance->GetConfig().EnableLogging)
			{
				_Context._Transport->_Logger.BeginLine("RX", method);
				_Context._Transport->_Logger.Append(val);
				_Context._Transport->_Logger.EndLine();
			}
			return val;
		}
		catch (std::exception& e)
		{
			if (_Context._Instance->GetConfig().EnableLogging)
			{
				std::stringstream s;
				s << "Unexpected exception during rpc-call: " << e.what();
				_Context._Transport->_Logger.Write(s.str());
			}
			throw;
		}
	}
};
